<?php
declare(strict_types=1);

namespace Vicentaotv\\Component\\Vicentaotv\\Site\\Controller;

use Joomla\\CMS\\MVC\\Controller\\BaseController;
use Joomla\\CMS\\Factory;
use Joomla\\CMS\\Component\\ComponentHelper;
use Joomla\\CMS\\Response\\JsonResponse;

class VideoController extends BaseController
{
    /**
     * POST JSON: { prompt: string, workflow: string, workflow_json?: object }
     * If workflow_json provided, will be sent directly to ComfyUI /prompt.
     * Otherwise tries to load media/com_vicentaotv/workflows/<workflow>.
     */
    public function generate(): void
    {
        $app = Factory::getApplication();
        $input = $app->input;
        $data = json_decode($input->json->getRaw(), true) ?: [];
        $prompt = (string)($data['prompt'] ?? '');
        $workflowName = (string)($data['workflow'] ?? '');
        $workflowJson = $data['workflow_json'] ?? null;

        $params = ComponentHelper::getParams('com_vicentaotv');
        $comfyUrl = (string)$params->get('comfy_url');
        if (!$comfyUrl) {
            echo new JsonResponse(['success' => false, 'error' => 'ComfyUI URL não configurada (comfy_url).']);
            return;
        }

        // Load workflow JSON from file if not provided
        if (!$workflowJson) {
            if (!$workflowName) {
                echo new JsonResponse(['success' => false, 'error' => 'Nenhum workflow_json ou nome de workflow fornecido.']);
                return;
            }
            $path = JPATH_ROOT . '/media/com_vicentaotv/workflows/' . basename($workflowName);
            if (!is_file($path)) {
                echo new JsonResponse(['success' => false, 'error' => 'Workflow não encontrado: ' . $workflowName]);
                return;
            }
            $jsonStr = file_get_contents($path);
            $workflowJson = json_decode($jsonStr, true);
            if (!$workflowJson) {
                echo new JsonResponse(['success' => false, 'error' => 'Workflow JSON inválido em: ' . $workflowName]);
                return;
            }
        }

        // Optionally inject prompt into workflow if a known field exists
        if ($prompt) {
            // Attempt simple injection convention: find keys named "prompt" and set them
            $workflowJson = $this->injectPrompt($workflowJson, $prompt);
        }

        // Send to ComfyUI
        $clientId = bin2hex(random_bytes(8));
        $payload = [
            'client_id' => $clientId,
            'prompt' => $workflowJson,
        ];

        $response = $this->postJson(rtrim($comfyUrl, '/') . '/prompt', $payload);

        if ($response['error'] ?? false) {
            echo new JsonResponse(['success' => false, 'error' => $response['error']]);
            return;
        }

        echo new JsonResponse([
            'success' => true,
            'status'  => 'queued',
            'client_id' => $clientId,
            'server_response' => $response,
        ]);
    }

    /**
     * GET status by client_id (placeholder). In a real implementation, use WebSocket or history endpoint.
     */
    public function status(): void
    {
        $app = Factory::getApplication();
        $clientId = (string)$app->input->get('client_id');
        if (!$clientId) {
            echo new JsonResponse(['success' => false, 'error' => 'client_id obrigatório']);
            return;
        }
        echo new JsonResponse(['success' => true, 'status' => 'pending', 'client_id' => $clientId]);
    }

    private function injectPrompt(array $workflow, string $prompt): array
    {
        // Recursively set "prompt" keys to provided prompt (basic convention for example graphs)
        array_walk_recursive($workflow, function (&$value, $key) use ($prompt) {
            if ($key === 'prompt') {
                $value = $prompt;
            }
        });
        return $workflow;
    }

    private function postJson(string $url, array $payload): array
    {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        $body = curl_exec($ch);
        $err  = curl_error($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        if ($err) {
            return ['error' => $err, 'code' => $code];
        }
        $json = json_decode((string)$body, true);
        return is_array($json) ? $json : ['code' => $code, 'raw' => $body];
    }
}
